const os = require('os');
const fs = require('fs');
const path = require('path');

const DEFAULT_UNIX = '/opt/dotenv/';

// Helper function to detect the operating system
const getOperatingSystemType = () => {
  const platform = os.platform();
  switch (platform) {
    case 'darwin':
      return 'MacOS';
    case 'win32':
      return 'Windows';
    case 'linux':
      return 'Linux';
    default:
      throw new Error(`Unsupported platform: ${platform}`);
  }
};

// Helper function for Windows-specific path check
const findWindowsPath = (projectName) => {
  const alphabet = 'abcdefghijklmnopqrstuvwxyz'.split('');
  for (const letter of alphabet) {
    const winPath = `${letter}:/dotenv/${projectName}`;
    if (fs.existsSync(winPath)) {
      return winPath;
    }
  }
  return null; // No valid path found
};

// Main function to get the default project path
const getDefaultPath = (projectName, ...customPath) => {
  const detectedOs = getOperatingSystemType();

  if (!projectName) {
    throw new Error('Project name is required.');
  }
  switch (detectedOs) {
    case 'MacOS':
    case 'Linux': {
      const basePath = customPath[0] ? customPath[0] : DEFAULT_UNIX;
      return path.join(basePath, projectName);
    }
    case 'Windows': {
      if (customPath[0]) {
        return path.join(customPath[0], projectName);
      } else {
        const windowsPath = findWindowsPath(projectName);
        if (!windowsPath) {
          throw new Error('No valid Windows drive found for the path.');
        }
        return windowsPath;
      }
    }
    default:
      throw new Error(`Unsupported OS: ${detectedOs}`);
  }
};

// Export the functions for use in other files
module.exports = {
  getOperatingSystemType,
  getDefaultPath,
};
