package web.multitask.trismegistoservices.utils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.WriterException;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.qrcode.QRCodeWriter;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import org.apache.commons.io.FileUtils;
import org.jetbrains.annotations.NotNull;
import org.json.JSONObject;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.charset.StandardCharsets;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

@Service
public class CommonUtils {

    private final String FILE_FOLDER = System.getProperty("java.io.tmpdir");

    public static String multipartFileToBase64(MultipartFile file) throws IOException {
        byte[] byteArray = file.getBytes();
        return Base64.getEncoder().encodeToString(byteArray);
    }

    public File base64ToFile(String base64, String name) {
        try {
            byte[] byteArray = Base64.getDecoder().decode(base64);
            String folder = FILE_FOLDER + "/" + name;
            File outputFile = new File(folder);
            FileUtils.writeByteArrayToFile(outputFile, byteArray);
            return outputFile;
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return null;
        }
    }

    public String fileToBase64(File file) {
        try {
            byte[] byteArray = FileUtils.readFileToByteArray(file);
            return Base64.getEncoder().encodeToString(byteArray);
        } catch (Exception e) {
            System.out.println(e.getMessage());
            return null;
        }
    }

    public String byteToBase64(byte[] byteArray) {
        return Base64.getEncoder().encodeToString(byteArray);
    }

    public Resource byteToResource(byte[] byteArray, String name) {
        return new ByteArrayResource(byteArray) {
            @Override
            public String getFilename() {
                return name;
            }
        };
    }

    public Resource fileToResource(File file) {
        return new FileSystemResource(file);
    }

    public static Resource base64ToResource(String base64, String name) {
        byte[] byteArray = Base64.getDecoder().decode(base64);
        return new ByteArrayResource(byteArray) {
            @Override
            public String getFilename() {
                return name;
            }
        };
    }

    public static boolean deleteAllFiles() {
        try {
            String tmpFolder = System.getProperty("java.io.tmpdir");
            File folder = new File(tmpFolder);
            if (!folder.exists() || !folder.isDirectory()) {
                return false;
            }
            File[] files = folder.listFiles();
            if (files != null) {
                for (File file : files) {
                    if (file.isFile()) {
                        if (!file.delete()) {
                            System.err.println("Failed to delete file: " + file.getAbsolutePath());
                        }
                    }
                }
            }
            return true;
        } catch (SecurityException e) {
            System.err.println("Security exception occurred: " + e.getMessage());
            return false;
        }
    }

    @NotNull
    public static ByteArrayOutputStream getByteArrayOutputStream(java.io.File tempDir) throws IOException {
        ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();

        try (ZipOutputStream zipOut = new ZipOutputStream(byteArrayOutputStream)) {

            for (java.io.File file : tempDir.listFiles()) {
                try (FileInputStream fis = new FileInputStream(file)) {
                    ZipEntry zipEntry = new ZipEntry(file.getName());
                    zipOut.putNextEntry(zipEntry);

                    byte[] bytes = new byte[1024];
                    int length;
                    while ((length = fis.read(bytes)) >= 0) {
                        zipOut.write(bytes, 0, length);
                    }

                    zipOut.closeEntry();
                }
            }
        }
        return byteArrayOutputStream;
    }

    public static String minifyJson(String json) {
        try {
            ObjectMapper objectMapper = new ObjectMapper();
            JsonNode jsonNode = objectMapper.readTree(json);
            return objectMapper.writeValueAsString(jsonNode);
        } catch (Exception e) {
            return e.getMessage();
        }
    }

    public static String jsonToUrlEncoded(JSONObject json) {
        StringBuilder result = new StringBuilder();
        Iterator<String> keys = json.keys();

        while (keys.hasNext()) {
            String key = keys.next();
            String value = json.optString(key);
            result.append(key).append("=").append(value);
            if (keys.hasNext()) {
                result.append("&");
            }
        }
        return result.toString();
    }

    public static String getCurrentDateRFC1123() {
        SimpleDateFormat dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.ENGLISH);
        dateFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
        return dateFormat.format(new Date());
    }

    public static String readInputStream(InputStream in) throws IOException {
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        byte[] temp = new byte[2048];
        int bytesRead;
        while ((bytesRead = in.read(temp)) != -1) {
            buffer.write(temp, 0, bytesRead);
        }
        return buffer.toString(StandardCharsets.UTF_8.name());
    }

    public static String generateQRCodeBase64(String message, int width, int height) {
        try {
            QRCodeWriter qrCodeWriter = new QRCodeWriter();
            BitMatrix bitMatrix = qrCodeWriter.encode(message, BarcodeFormat.QR_CODE, width, height);
            BufferedImage qrImage = MatrixToImageWriter.toBufferedImage(bitMatrix);
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ImageIO.write(qrImage, "PNG", baos);
            byte[] imageBytes = baos.toByteArray();
            return Base64.getEncoder().encodeToString(imageBytes);
        } catch (WriterException | java.io.IOException e) {
            System.out.println(e.getMessage());
            return null;
        }
    }

}