package web.multitask.trismegistoservices.rest;

import org.json.JSONObject;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.web.bind.annotation.*;
import web.multitask.trismegistoservices.model.User;
import web.multitask.trismegistoservices.repository.UserRepository;
import web.multitask.trismegistoservices.utils.tokenUtils;

import java.math.BigInteger;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Objects;

import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;

@RestController
@RequestMapping("/token")
@CrossOrigin
class tokenRest {

    private final UserRepository userRepo = new UserRepository();

    private final tokenUtils jwtTokenUtil;
    private final web.multitask.trismegistoservices.singleton.tokenSingleton tokenSingleton;

    public tokenRest(tokenUtils jwtTokenUtil, web.multitask.trismegistoservices.singleton.tokenSingleton tokenSingleton) {
        this.jwtTokenUtil = jwtTokenUtil;
        this.tokenSingleton = tokenSingleton;
    }

    @PostMapping("/database")
    public ResponseEntity<?> setDatabase(@RequestBody String token) {
        JSONObject json = new JSONObject(token);
        if (json.has("db") && json.has("user") && json.has("password") && json.has("url")) {
            String dataToken = jwtTokenUtil.generateDataSource(json);
            return ResponseEntity.ok(new JSONObject().put("token", dataToken).put("message", "Generated").put("status", true).toMap());
        } else {
            return ResponseEntity.status(400).body(new JSONObject().put("token", "").put("message", "Invalid Data").put("status", false).toMap());
        }
    }

    @PostMapping("/authenticate")
    public ResponseEntity<?> createAuthenticationToken(@RequestBody String authenticationRequest) {
        JSONObject response;
        JSONObject json = new JSONObject(authenticationRequest);
        String username = json.getString("username");
        try {
            UserDetails userDetails = userRepo.findByUsername(username);
            if (!Objects.equals(userDetails.getPassword(), json.getString("password"))) {
                response = new JSONObject().put("token", "").put("message", "Invalid Credentials").put("status", false);
                return ResponseEntity.status(401).body(response.toMap());
            } else {
                boolean onelife = json.optBoolean("onelife", false);
                String generatedToken = jwtTokenUtil.generateToken((User) userDetails, json.optBigInteger("ms", onelife ? BigInteger.valueOf(0) : BigInteger.valueOf(3600000)), onelife);
                if (onelife) {
                    tokenSingleton.addToken(generatedToken);
                }
                return ResponseEntity.ok(new JSONObject().put("token", generatedToken).put("message", "Generated").put("status", true).toMap());
            }
        } catch (Exception e) {
            response = new JSONObject().put("token", "").put("message", "Invalid Credentials").put("status", false);
            return ResponseEntity.status(401).body(response.toMap());
        }
    }

    @PostMapping("/validate")
    public ResponseEntity<?> validateToken(@RequestBody String token) {
        JSONObject json = new JSONObject(token);
        JSONObject response = new JSONObject();

        boolean isTokenExpired = jwtTokenUtil.isTokenExpired(json.getString("token"));
        if (isTokenExpired) {
            response.put("message", "Token expired").put("status", false);
            return ResponseEntity.status(403).body(response.toMap());
        }

        boolean oneLifeToken = tokenSingleton.isTokenAvailable(json.getString("token"));

        if (oneLifeToken) {
            response.put("message", "El token de una vida sigue en pie").put("status", true);
            return ResponseEntity.ok(response.toMap());
        }

        boolean validToken = jwtTokenUtil.validateToken(json.getString("token"));
        if (!validToken) {
            response.put("message", "El token no existe").put("status", false);
            return ResponseEntity.status(401).body(response.toMap());
        }

        String dataToken = jwtTokenUtil.getDataToken(json.getString("token"));
        if (dataToken == null) {
            response.put("message", "Invalid Token").put("status", false);
            return ResponseEntity.status(401).body(response.toMap());
        }

        response.put("message", "Valid Token").put("status", true);
        return ResponseEntity.ok(response.toMap());
    }

    @PostMapping("/service/authenticate")
    public ResponseEntity<?> generateToken(@RequestBody String token) {
        JSONObject json = new JSONObject(token);
        UserDetails userDetails = userRepo.findByUsername(json.getString("username"));
        if (userDetails == null) {
            return ResponseEntity.status(401).body(new JSONObject().put("token", "").put("message", "Credenciales invalidas").put("status", false).toMap());
        } else if (userDetails.getAuthorities().stream().anyMatch(a -> a.getAuthority().equals("SERVICE"))) {
            boolean onelife = json.optBoolean("onelife", false);
            String generatedToken = jwtTokenUtil.generateToken((User) userDetails, json.optBigInteger("ms", onelife ? BigInteger.valueOf(0) : BigInteger.valueOf(3600000)), false);
            if (onelife) {
                tokenSingleton.addToken(generatedToken);
            }
            return ResponseEntity.ok(new JSONObject().put("token", generatedToken).put("message", "Generado").put("status", true).toMap());
        } else {
            return ResponseEntity.status(401).body(new JSONObject().put("token", "").put("message", "Credenciales invalidas").put("status", false).toMap());
        }
    }

    @PostMapping("/remaining")
    public ResponseEntity<?> remainingTime(@RequestBody String token) {
        JSONObject json = new JSONObject(token);
        try {
            Long remaining = jwtTokenUtil.getExperyTime(json.getString("token"));
            Date expirationDate = new Date(System.currentTimeMillis() + remaining);
            SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
            return ResponseEntity.ok(new JSONObject().put("remaining", remaining).put("message", "OK").put("expiration", dateFormat.format(expirationDate)).put("status", true).toMap());
        } catch (Exception e) {
            return ResponseEntity.status(401).body(new JSONObject().put("remaining", 0).put("message", "Token invalido").put("status", false).toMap());
        }
    }

    @PostMapping("/tokenize")
    public ResponseEntity<?> tokenize(@RequestBody String data) {
        JSONObject json = new JSONObject(data);
        try {
            boolean onelife = json.optBoolean("onelife", false);
            BigInteger ms = json.optBigInteger("ms", BigInteger.valueOf(3600000));
            if (json.optBoolean("onelife", false)) {
                json.remove("onelife");
            }
            if (json.has("ms")) {
                json.remove("ms");
            }
            String tokenized = jwtTokenUtil.tokenizeData(json.toString(), ms, onelife);
            if (onelife) {
                tokenSingleton.addToken(tokenized);
            }
            return ResponseEntity.ok(new JSONObject().put("token", tokenized).put("message", "OK").put("status", true).toMap());
        } catch (Exception e) {
            return ResponseEntity.status(400).body(new JSONObject().put("token", "").put("message", e.getMessage()).put("status", false).toMap());
        }
    }

    @PostMapping("/detokenize")
    public ResponseEntity<?> detokenize(@RequestBody String tokenBody) {
        JSONObject json = new JSONObject(tokenBody);
        String token = json.getString("token");
        boolean doConsume = json.optBoolean("consume", false);

        try {

            String detokenized = jwtTokenUtil.detokenizeData(token);
            if (detokenized.isEmpty()) {
                return ResponseEntity.status(498).body(
                        new JSONObject()
                                .put("data", "")
                                .put("message", "Token inválido")
                                .put("status", false)
                                .toMap()
                );
            }

            if (jwtTokenUtil.isTokenExpired(token)) {
                return ResponseEntity.status(419).body(
                        new JSONObject()
                                .put("data", "")
                                .put("message", "Token ha expirado")
                                .put("status", false)
                                .toMap()
                );
            }

            if (!tokenSingleton.isTokenAvailable(token)) {
                return ResponseEntity.status(419).body(
                        new JSONObject()
                                .put("data", "")
                                .put("message", "Token ya ha sido consumido")
                                .put("status", false)
                                .toMap()
                );
            }
            if (doConsume) {
                tokenSingleton.consumeToken(token);
            }
            return ResponseEntity.ok(
                    new JSONObject()
                            .put("data", detokenized)
                            .put("message", "OK")
                            .put("status", true)
                            .toMap()
            );
        } catch (Exception e) {
            return ResponseEntity.status(500).body(
                    new JSONObject()
                            .put("data", "")
                            .put("message", e.getMessage())
                            .put("status", false)
                            .toMap()
            );
        }
    }


    @PostMapping("/consume")
    public ResponseEntity<?> consumeToken(@RequestBody String token) {
        JSONObject json = new JSONObject(token);
        try {
            if (tokenSingleton.consumeToken(json.getString("token"))) {
                return ResponseEntity.ok(new JSONObject().put("message", "OK").put("status", true).toMap());
            } else {
                return ResponseEntity.status(400).body(new JSONObject().put("message", "Token invalido").put("status", false).toMap());
            }
        } catch (Exception e) {
            return ResponseEntity.status(400).body(new JSONObject().put("message", e.getMessage()).put("status", false).toMap());
        }
    }
}