package web.multitask.trismegistoservices.services;

import java.awt.Color;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCell;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFColor;
import org.apache.poi.xssf.usermodel.XSSFRow;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.apache.poi.xssf.usermodel.extensions.XSSFCellBorder.BorderSide;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Service;

@Service
public class ExcelService {

    public byte[] generateExcel(JSONObject json, int size) throws IOException {
        ByteArrayOutputStream outByteStream = new ByteArrayOutputStream();
        XSSFWorkbook workbook = new XSSFWorkbook();
        createSheet(workbook, json, size);
        try {
            workbook.write(outByteStream);
            workbook.close();
            outByteStream.close();
            outByteStream.flush();
            return outByteStream.toByteArray();
        } catch (Exception e) {
            workbook.close();
            outByteStream.close();
            outByteStream.flush();
            System.out.println(e.getMessage());
            return null;
        }
    }

    public void createSheet(XSSFWorkbook workbook, JSONObject json, int sizeOfBytes) {
        AtomicReference<List<CompletableFuture<Boolean>>> results = new AtomicReference<>(new ArrayList<>());
        if (json.optJSONArray("sheetArray", new JSONArray()).isEmpty()) {
            XSSFSheet sheet = workbook.createSheet(json.optString("sheet_name", "no_name"));
            buildSheet(workbook, sheet, json);
            JSONArray identifiers = json.getJSONArray("identifiers");
            for (int i = 0; i < identifiers.length(); i++) {
                if (json.optBoolean("auto_adjustment", true)) {
                     findLargerTextByIdentifer(sheet, json.getJSONArray("data"), identifiers);
                }
            }
           for (int j = 0; j < json.optJSONArray("columns_size", new JSONArray()).length(); j++) {
                if (json.getJSONArray("columns_size").getInt(j) > 0) {
                    sheet.setColumnWidth(j, json.getJSONArray("columns_size").getInt(j));
                }
            }
        } else {
            JSONArray sheetArray = json.optJSONArray("sheetArray", new JSONArray());
            for (int i = 0; i < sheetArray.length(); i++) {
                JSONObject jsonObject = sheetArray.getJSONObject(i);
                XSSFSheet sheet = workbook.createSheet(jsonObject.optString("sheet_name", "no_name"));
                JSONArray identifiers = jsonObject.getJSONArray("identifiers");
                for (int j = 0; j < identifiers.length(); j++) {
                    if (jsonObject.optBoolean("auto_adjustment", true)) {
                        findLargerTextByIdentifer(sheet, jsonObject.getJSONArray("data"), identifiers);
                    }
                }
                buildSheet(workbook, sheet, jsonObject);
                if (!jsonObject.optBoolean("auto_adjustment", true)) {
                    for (int k = 0; k < jsonObject.getJSONArray("columns_size").length(); k++) {
                        sheet.autoSizeColumn(k);
                    }
                }
                for (int j = 0; j < jsonObject.optJSONArray("columns_size", new JSONArray()).length(); j++) {
                    if (jsonObject.getJSONArray("columns_size").getInt(j) > 0) {
                        sheet.setColumnWidth(j, jsonObject.getJSONArray("columns_size").getInt(j));
                    }
                }
            }
        }
    }

    private void findLargerTextByIdentifer(XSSFSheet sheet, JSONArray data, JSONArray identifiers) {
        int[] maxLengthByIdentifier = new int[identifiers.length()];
        for (int i = 0; i < data.length(); i++) {
            JSONObject row_data = data.optJSONObject(i);
            for (int j = 0; j < identifiers.length(); j++) {
                int length = row_data.optString(identifiers.optString(j, "")).length();
                int length2 = row_data.optString(String.valueOf(j), "").length();
                if (length > maxLengthByIdentifier[j] || length2 > maxLengthByIdentifier[j]) {
                    maxLengthByIdentifier[j] = Math.max(length, length2);
                }
            }
        }

        for (int i = 0; i < maxLengthByIdentifier.length; i++) {
            sheet.setColumnWidth(i, ((maxLengthByIdentifier[i] * 375)));
        }
    }

    public void buildSheet(XSSFWorkbook workbook, XSSFSheet sheet, JSONObject json) {

        final XSSFCellStyle defaultStyle = createCellStyle(workbook, new JSONObject());
        List<String> stylesSaved = new ArrayList<>();
        List<XSSFCellStyle> styleObjectsSaved = new ArrayList<>();

        XSSFRow row;
        XSSFCell cell;
        XSSFCellStyle cellStyle;

        JSONArray identifiers = json.optJSONArray("identifiers", new JSONArray());
        JSONArray data = json.optJSONArray("data", new JSONArray());
        JSONArray styles = json.optJSONArray("styles", new JSONArray());

        fillStylesArray(styles, workbook, stylesSaved, styleObjectsSaved);
        int i = 0;

        if (json.has("title")) {
            row = sheet.createRow(i);
            JSONObject titleJson = json.getJSONObject("title");
            JSONObject style = titleJson.optJSONObject("style", new JSONObject());
            cell = row.createCell(0);
            cell.setCellValue(titleJson.optString("text", ""));
            if (style.isEmpty()) {
                cell.setCellStyle(defaultStyle);
            } else {
                if (!stylesSaved.contains(style.toString())) {
                    stylesSaved.add(style.toString());
                    cellStyle = createCellStyle(workbook, style);
                    cell.setCellStyle(cellStyle);
                    styleObjectsSaved.add(cellStyle);
                } else {
                    cell.setCellStyle(styleObjectsSaved.get(stylesSaved.indexOf(style.toString())));
                }
            }
            sheet.addMergedRegion(new CellRangeAddress(i, i, 0, style.getInt("colspan") - 1));
            i++;
        }

        for (; i < data.length(); i++) {
            int index = json.has("title") ? i - 1 : i;
            row = sheet.createRow(i);
            JSONObject row_data = data.optJSONObject(index);
            for (int j = 0; j < identifiers.length(); j++) {
                cell = row.createCell(j);
                JSONArray styleArray = styles.optJSONArray(index, new JSONArray());
                JSONObject style = styleArray.optJSONObject(j, new JSONObject());
                String value = row_data.optString(identifiers.getString(j), row_data.optString(String.valueOf(j), ""));

                if (style.isEmpty()) {
                    cell.setCellStyle(defaultStyle);
                    cell.setCellValue(value);
                } else {
                    if (!stylesSaved.contains(style.toString())) {
                        stylesSaved.add(style.toString());
                        cellStyle = createCellStyle(workbook, style);
                        cell.setCellStyle(cellStyle);
                        styleObjectsSaved.add(cellStyle);
                        cell.setCellValue(value);
                    } else {
                        cell.setCellStyle(styleObjectsSaved.get(stylesSaved.indexOf(style.toString())));
                        cell.setCellValue(value);
                    }
                    if (style.has("format")) {
                        setCellFormat(cell, style.optString("format", "text"), value);
                    }
                    if (style.has("colspan")) {
                        sheet.addMergedRegion(new CellRangeAddress(i, i, j, j + style.optInt("colspan", 0)));
                    }
                }

            }
        }

    }

    public void setCellFormat(XSSFCell cell, String format, String value) {
        switch (format) {
            case "number":
                cell.setCellType(CellType.NUMERIC);
                XSSFCellStyle style = cell.getCellStyle();
                DataFormat dataFormat = cell.getSheet().getWorkbook().createDataFormat();
                style.setDataFormat(dataFormat.getFormat("0.00"));
                cell.setCellStyle(style);
                cell.setCellValue(Double.parseDouble(value));
                break;
            case "date":
                cell.setCellType(CellType.NUMERIC);
                cell.setCellValue(value);
                break;
            case "formula":
                cell.setCellType(CellType.FORMULA);
                cell.setCellFormula(value);
                break;
            default:
                cell.setCellType(CellType.STRING);
                cell.setCellValue(value);
                break;
        }
    }

    private XSSFCellStyle createCellStyle(XSSFWorkbook workbook, JSONObject styleJson) {

        XSSFColor background = getXSSFColor(styleJson.optString("background", "#ffffff"));
        XSSFColor foreground = getXSSFColor(styleJson.optString("foreground", styleJson.optString("color", "#333333")));
        boolean bold = styleJson.optBoolean("bold", false);
        boolean border = styleJson.optBoolean("border", true);
        String align = styleJson.optString("textAlign", "center");
        XSSFCellStyle style = workbook.createCellStyle();
        Font font = workbook.createFont();
        style.setAlignment(align.equals("center") ? HorizontalAlignment.CENTER : align.equals("left") ? HorizontalAlignment.LEFT : HorizontalAlignment.RIGHT);
        style.setWrapText(styleJson.optBoolean("wrapText", false));
        style.setVerticalAlignment(VerticalAlignment.CENTER);
        style.setFillPattern(FillPatternType.SOLID_FOREGROUND);
        style.setBorderBottom(BorderStyle.THIN);
        style.setBorderTop(BorderStyle.THIN);
        style.setBorderLeft(BorderStyle.THIN);
        style.setBorderRight(BorderStyle.THIN);

        if (border) {
            style.setBorderColor(BorderSide.BOTTOM, getXSSFColor("#333333"));
            style.setBorderColor(BorderSide.TOP, getXSSFColor("#333333"));
            style.setBorderColor(BorderSide.LEFT, getXSSFColor("#333333"));
            style.setBorderColor(BorderSide.RIGHT, getXSSFColor("#333333"));
        } else {
            style.setBorderBottom(BorderStyle.THIN);
            style.setBorderTop(BorderStyle.THIN);
            style.setBorderLeft(BorderStyle.THIN);
            style.setBorderRight(BorderStyle.THIN);
            style.setBorderColor(BorderSide.BOTTOM, getXSSFColor("#aaaaaa"));
            style.setBorderColor(BorderSide.TOP, getXSSFColor("#aaaaaa"));
            style.setBorderColor(BorderSide.LEFT, getXSSFColor("#aaaaaa"));
            style.setBorderColor(BorderSide.RIGHT, getXSSFColor("#aaaaaa"));
        }

        font.setBold(bold);
        style.setFont(font);

        if (background != null) {
            style.setFillForegroundColor(background);
        }

        if (foreground != null) {
            style.getFont().setColor(foreground);
        }

        return style;
    }

    public XSSFColor getXSSFColor(String rgb) {
        int red = Integer.parseInt(rgb.substring(1, 3), 16);
        int green = Integer.parseInt(rgb.substring(3, 5), 16);
        int blue = Integer.parseInt(rgb.substring(5, 7), 16);
        Color awtColor = new Color(red, green, blue);
        XSSFColor xssfColor = new XSSFColor();
        xssfColor.setRGB(new byte[]{(byte) awtColor.getRed(), (byte) awtColor.getGreen(), (byte) awtColor.getBlue()});
        return xssfColor;
    }

    public Resource convertByteToResource(byte[] data, String filename) {
        return new ByteArrayResource(data) {
            @Override
            public String getFilename() {
                return filename;
            }
        };
    }

    public void fillStylesArray(JSONArray styles, XSSFWorkbook workbook, List<String> stylesSaved, List<XSSFCellStyle> styleObjectsSaved) {
        for (int i = 0; i < styles.length(); i++) {
            JSONArray styleArray = styles.optJSONArray(i, new JSONArray());
            for (int j = 0; j < styleArray.length(); j++) {
                JSONObject style = styleArray.optJSONObject(j, new JSONObject());
                if (!stylesSaved.contains(style.toString())) {
                    stylesSaved.add(style.toString());
                    XSSFCellStyle cellStyle = createCellStyle(workbook, style);
                    styleObjectsSaved.add(cellStyle);
                }
            }
        }
    }
}